/*
 * Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * SPDX-License-Identifier:  GPL-2.0+
 */

#include <inttypes.h>
#include "unlocker.h"

// This structure is what gets hashed to produce a challenge hash
typedef struct
{
    pseudo_uint64_t salt;
    uint32_t      relock_boot_count;
    unique_data_t unique_data;
} challenge_in_t;


typedef struct
{
    hash_t       challenge_hash; // hex value
    unique_tag_t challenge_tag;  // tag is an ascii string
} challenge_out_t;

#ifndef MIN
#define MIN(a, b) ((a) < (b) ? (a) : (b))
#endif

static unlock_block_t static_unlock_block;
/* Ideally the challenge str would be the full hash with the DSN appended.
   However we're limited by the length of data which can be output over fastboot,
   so we read the actual usable length from the data block.  We always overlay the
   DSN over the end of the data so we need the str to be at least that long, though
   we want *some* random hash in the string, so enforce a minimum amount.  Also it
   can't be longer than hash + DSN length, which is the space we allocate. */
#define static_challenge_str_len (sizeof(hash_t)*2 + sizeof(unique_tag_t) + 1)
#define MIN_HASH_LEN_USED 16
#define MAX_CHALLENGE_STR_LEN (static_challenge_str_len)
#define MIN_CHALLENGE_STR_LEN (sizeof(unique_tag_t) + MIN_HASH_LEN_USED + 1)

static char static_challenge_str[static_challenge_str_len] __attribute__((aligned (16)));


static bootloader_status_t make_unlock_challenge(uint32_t relock_boot_count,
                                                 pseudo_uint64_t *challenge_salt,
                                                 unique_data_t   *unique_data,
                                                 challenge_out_t *challenge_out);
static int get_challenge_str(challenge_out_t *challenge,
                             char *challenge_str,
                             int challenge_str_len);

/*
 * Retrieve the unlocked/locked status
 *
 * @return BOOTLOADER_OK if successful
 */
bootloader_status_t is_unlocked(unlock_state_t *unlocked_out)
{
    bool                secured_status;
    unlock_block_t      unlock_block;
    bootloader_status_t status;
    hash_t              hashed_challenge;
    challenge_out_t     challenge;
    unique_data_t       unique_data;

    status = platform_get_unlock_state(unlocked_out);
    if (status != BOOTLOADER_OK)
    {
        boot_debug("platform_get_unlock_state Failed\n");
        goto out;
    }
    else if (*unlocked_out > UNLOCK_STATE_INVALID)
    {
        boot_debug("Unlock state already known (%" PRIu32 ")\n", *unlocked_out);
        goto out;
    }

    status = platform_get_secured_status(&secured_status);
    if( status != BOOTLOADER_OK )
    {
        boot_debug("platform_get_secured_status Failed\n");
        goto out;
    }

    if (!secured_status)
    {
        boot_debug("Device not yet secured - treating as unlocked\n");
        *unlocked_out = UNLOCK_STATE_NON_PROD;
        status = BOOTLOADER_OK;
        goto out;
    }

    // Check for unlock signature
    status = platform_get_unlock_details(&unlock_block);
    if (status != BOOTLOADER_OK)
    {
        boot_debug("platform_get_unlock_details failed\n");
        goto out;
    }

    if (unlock_block.is_set != UNLOCK_BLOCK_SET)
    {
        boot_debug("No unlock signature present\n");
        *unlocked_out = UNLOCK_STATE_LOCKED;
        status = BOOTLOADER_OK;
        goto out;
    }

    boot_debug("An unlock signature exists - verifying\n");
    status = platform_get_device_unique_data( &unique_data );
    if (status != BOOTLOADER_OK) {
        boot_debug("%s: platform_get_device_unique_data failed!\n", __func__);
        goto out;
    }

    status = make_unlock_challenge(unlock_block.relock_boot_count,
                                   &unlock_block.unlock_challenge_salt,
                                   &unique_data,
                                   &challenge);
    if (status != BOOTLOADER_OK)
    {
        boot_debug("make_unlock_challenge failed\n");
        goto out;
    }
    memset(static_challenge_str, 0, static_challenge_str_len);
    status = get_challenge_str(&challenge, static_challenge_str, unlock_block.challenge_str_len);
    if (status != BOOTLOADER_OK)
    {
        boot_debug("get_challenge_str failed!\n");
        goto out;
    }
    boot_debug("Unlock challenge is: %s\n", static_challenge_str);

    status = platform_hash( &hashed_challenge, static_challenge_str, unlock_block.challenge_str_len-1 );
    if (status != BOOTLOADER_OK)
    {
        boot_debug("platform_hash failed!\n");
        goto out;
    }

    boot_debug_buf("Unlock challenge hash is: ", &hashed_challenge, sizeof(hashed_challenge));

    if ( BOOTLOADER_VERIFY_SUCCESS == platform_verify_signature( &hashed_challenge, &unlock_block.unlock_signature, BOOT_COUNTED_UNLOCK_PUBLIC_KEY ) )
    {
        uint32_t boot_count;

        boot_debug("Signature is correct for boot-count restricted unlock\n");

        status = platform_get_current_boot_count(&boot_count);
        if (status != BOOTLOADER_OK)
        {
            boot_debug("platform_get_current_boot_count failed!\n");
            goto out;
        }

        if ((boot_count < unlock_block.relock_boot_count - RESTRICTED_MAX_BOOTS) ||
            (boot_count >=  unlock_block.relock_boot_count))
        {
            // Boot count is either invalid or expired
            boot_debug("Bad boot count %" PRIu32 " - relocking\n", boot_count);
            status = relock(true);
            if (status != BOOTLOADER_OK)
            {
                boot_debug("relock failed!\n");
                goto out;
            }
            status = BOOTLOADER_BOOT_COUNT_EXPIRED;
            goto out;
        }

        // Boot count valid.
        boot_debug("Valid boot count %" PRIu32 " - UNLOCKED\n", boot_count);
        *unlocked_out = UNLOCK_STATE_RESTRICTED_UNLOCK;

        status = BOOTLOADER_OK;
        goto out;
    }

    if ( BOOTLOADER_VERIFY_SUCCESS == platform_verify_signature( &hashed_challenge, &unlock_block.unlock_signature, UNRESTRICTED_UNLOCK_PUBLIC_KEY ) )
    {
        boot_debug("Signature is correct for unrestricted unlock - UNLOCKED\n");
        *unlocked_out = UNLOCK_STATE_FULL_UNLOCK;

        status = BOOTLOADER_OK;
        goto out;
    }

    // Unknown unlock signature - wipe it
    boot_debug("Signature is invalid - relocking\n");

    status = relock(true);
    if (status != BOOTLOADER_OK)
    {
        boot_debug("relock failed!\n");
        goto out;
    }
out:
    if (status != BOOTLOADER_OK)
        *unlocked_out = UNLOCK_STATE_INVALID;
    platform_set_unlock_state(*unlocked_out);
    return status;
}

/*
 * Generate a challenge
 *
 * @return BOOTLOADER_OK if successful
 */
static int get_challenge_str(challenge_out_t *challenge,
                             char *challenge_str,
                             int challenge_str_len)
{
    static const char hex_vals[16] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };

    /* Range check externally-supplied challenge_str_len */
    if ((challenge_str_len > MAX_CHALLENGE_STR_LEN) || (challenge_str_len < MIN_CHALLENGE_STR_LEN))
    {
        return BOOTLOADER_OTHER_ERROR;
    }

    int chall_hash_bytes = MIN(sizeof(challenge->challenge_hash) * 2, challenge_str_len - 1) - sizeof(challenge->challenge_tag);

    memset(challenge_str, 0, challenge_str_len);

    for (int i = 0; i < chall_hash_bytes; i+=2)
    {
        challenge_str[i]   = ( hex_vals[(challenge->challenge_hash.value[i>>1] >> 4) ] );
        challenge_str[i+1] = ( hex_vals[(challenge->challenge_hash.value[i>>1] & 0xf)] );
    }

    memcpy(challenge_str + chall_hash_bytes, challenge->challenge_tag.value, sizeof(challenge->challenge_tag));
    return BOOTLOADER_OK;
}

static bootloader_status_t make_unlock_challenge(uint32_t relock_boot_count,
                                                 pseudo_uint64_t *challenge_salt,
                                                 unique_data_t   *unique_data,
                                                 challenge_out_t *challenge_out)
{
    challenge_in_t challenge_data;
    bootloader_status_t ret;

    challenge_data.relock_boot_count = relock_boot_count;
    memcpy(challenge_data.salt.value, challenge_salt->value, sizeof(pseudo_uint64_t));
    memcpy(challenge_data.unique_data.value, unique_data->value, sizeof(unique_data_t));

    ret = platform_hash(&challenge_out->challenge_hash, (uint8_t*)&challenge_data, sizeof(challenge_data));
    if (ret != BOOTLOADER_OK) {
        boot_debug("%s: platform_hash failed!\n", __func__);
        goto out;
    }
    ret = platform_get_device_unique_tag(&challenge_out->challenge_tag);
    if (ret != BOOTLOADER_OK) {
        boot_debug("%s: platform_get_device_unique_tag failed!\n", __func__);
        goto out;
    }

out:
    return ret;
}


/*
 * Re-lock the device
 * Wipes the unlock signature and challenge details
 *
 * @return BOOTLOADER_OK if successful
 */
bootloader_status_t relock(bool reboot)
{
    bootloader_status_t         status;

    static_unlock_block.is_set = 0;

    status = platform_relock();
    if (status != BOOTLOADER_OK )
    {
        boot_debug("platform_relock failed!\n");
        return status;
    }

    if (reboot)
        platform_reboot();

    return BOOTLOADER_OK;
}

int unlocker_relock_device(bool reboot)
{
    bootloader_status_t status = relock(reboot);

    if (status != BOOTLOADER_OK)
        return -1;
    else
        return 0;
}

int unlocker_get_challenge(char *unlock_code, int unlock_code_len)
{
    unique_data_t       unique_data;
    challenge_out_t     challenge;
    bootloader_status_t status;

    /* Range check unlock_code_len to ensure it will pass the same
       check we use on retrieval */
    if ((unlock_code_len > MAX_CHALLENGE_STR_LEN) ||
        (unlock_code_len < MIN_CHALLENGE_STR_LEN))
    {
        return -1;
    }

    if (static_unlock_block.is_set != UNLOCK_BLOCK_IN_MEM)
    { /* Always generate the same challenge for a given boot */
        memset(&static_unlock_block, 0, sizeof(unlock_block_t));
        status = platform_get_current_boot_count(&static_unlock_block.relock_boot_count);
        if (status != BOOTLOADER_OK)
        {
            boot_debug("%s: platform_get_current_boot_count failed!\n", __func__);
            return -1;
        }
        /* This is a new challenge; make sure the boot count is invalid for temp
           unlock codes */
        static_unlock_block.relock_boot_count += RESTRICTED_MAX_BOOTS;

        status = platform_get_device_unique_data( &unique_data );
        if (status != BOOTLOADER_OK) {
            boot_debug("%s: platform_get_device_unique_data failed!\n", __func__);
            return -1;
        }
        platform_random(&static_unlock_block.unlock_challenge_salt);
        make_unlock_challenge(static_unlock_block.relock_boot_count,
                              &static_unlock_block.unlock_challenge_salt,
                              &unique_data,
                              &challenge);
        /* length of the string to be sent to codesigner */
        static_unlock_block.challenge_str_len = unlock_code_len;

        if (get_challenge_str(&challenge,
                              static_challenge_str,
                              static_unlock_block.challenge_str_len) != 0) {
            boot_debug("%s: get_challenge_str failed!\n", __func__);
            return -1;
        }
        /* Mark the data as valid in memory.  We won't commit it to flash until
           we get a signed unlock code returned by codesigner */
        static_unlock_block.is_set = UNLOCK_BLOCK_IN_MEM;
    }
    memcpy(unlock_code, static_challenge_str, unlock_code_len);

    return 0;
}

int unlocker_write_signature(uint8_t *signature, int signature_len)
{
    hash_t              challenge_hash;
    bootloader_status_t status;

    /* Writing a new signature means we need to invalidate the current
       unlock state */
    platform_set_unlock_state(UNLOCK_STATE_INVALID);

    if (static_unlock_block.is_set != UNLOCK_BLOCK_IN_MEM)
    {
        boot_debug("%s: invalid unlock block!\n", __func__);
        return -1;
    }

    if (signature_len != sizeof(static_unlock_block.unlock_signature))
    {
        boot_debug("%s: Unexpected signature length (%" PRIu32 " vs %" PRIu32 ")!\n", __func__,
                   signature_len, sizeof(static_unlock_block.unlock_signature));
        return -1;
    }
    memcpy(static_unlock_block.unlock_signature.value, signature, sizeof(static_unlock_block.unlock_signature));

    status = platform_hash( &challenge_hash, static_challenge_str, static_unlock_block.challenge_str_len-1 );
    if (status != BOOTLOADER_OK )
    {
        boot_debug("platform_hash failed!\n");
        return -1;
    }
    boot_debug("Unlock challenge is: %s\n", static_challenge_str);
    boot_debug_buf("Unlock challenge hash is: ", challenge_hash.value, sizeof(challenge_hash.value));

    if ( ( BOOTLOADER_VERIFY_SUCCESS != platform_verify_signature( &challenge_hash, &static_unlock_block.unlock_signature, BOOT_COUNTED_UNLOCK_PUBLIC_KEY ) ) &&
         ( BOOTLOADER_VERIFY_SUCCESS != platform_verify_signature( &challenge_hash, &static_unlock_block.unlock_signature, UNRESTRICTED_UNLOCK_PUBLIC_KEY ) ) )
    {
        boot_debug("Signature does not verify - relocking!\n");
        relock(true);
        return -1;
    }

    // Valid signature - Save it
    static_unlock_block.is_set = UNLOCK_BLOCK_SET;
    status = platform_write_unlock_details( &static_unlock_block );
    if (status != BOOTLOADER_OK )
    {
        boot_debug("platform_write_unlock_details failed!\n");
        return -1;
    }
    return 0;
}


/* Factory test functionality. Only loosely related to unlock.  It is here because;
    - It stores data in the unlock flash block
    - We want to wipe it on device lock
 */
int enable_factory_test_mode(uint32_t factory_test_port)
{
    unlock_state_t unlock_state;
    bootloader_status_t status;
    int err = -1;

    is_unlocked(&unlock_state);
    if (unlock_state >= UNLOCK_STATE_RESTRICTED_UNLOCK) {
        status = platform_set_factory_test_mode(true, factory_test_port);
        if (status == BOOTLOADER_OK) {
            err = 0;
        }
    }

    return err;
}

int disable_factory_test_mode(void)
{
    unlock_state_t unlock_state;
    bootloader_status_t status;
    int err = -1;

    is_unlocked(&unlock_state);
    if (unlock_state >= UNLOCK_STATE_RESTRICTED_UNLOCK) {
        status = platform_set_factory_test_mode(false, 0);
        if (status == BOOTLOADER_OK) {
            err = 0;
        }
    } else {
        /* not an error, but nothing to do - being locked implies disabled */
        err = 0;
    }
    return err;
}

/* Get the factory test mode state and port.
 * enabled and port are only valid if return code is 0
 */
int get_factory_test_mode(bool *enabled, uint32_t *port)
{
    unlock_state_t unlock_state;
    int err = -1;

    if ((enabled == NULL) || (port == NULL))
        return err;

    is_unlocked(&unlock_state);
    if (unlock_state >= UNLOCK_STATE_RESTRICTED_UNLOCK) {
        if (platform_get_factory_test_mode(enabled, port) == BOOTLOADER_OK) {
            err = 0;
        }
    } else {
        err = 0;
        *enabled = false;
        *port = 0;
    }
    return err;
}
